
#include <stdio.h>
#include "miracl.h"
#include <stdlib.h>
#include <string.h>

static miracl *mip;

void ecsbkeygen(char* parameter_file)
{
    long seed=11945;
    FILE *fp;
    int i,ep,m[4];
    epoint *g,*w;
    big n[4],a,b,q,x,y,d;
    fp=fopen(parameter_file,"r");
    if (fp==NULL)
    {
        printf("parameter file does not exist\n");
        return;
    }

    mip=mirsys(150,0);
    a=mirvar(1);
    b=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    d=mirvar(0);
    for(i=0;i<4;i++) n[i]=mirvar(0);
    mip->IOBASE=10;    
    for(i=0;i<4;i++)
    {
      cinnum(n[i],fp);
      m[i]=size(n[i]);
    }
    mip->IOBASE=16;
    cinnum(b,fp);
    cinnum(x,fp);
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);
    mip->IOBASE=16;
    
    fclose(fp);
    printf("\nInitializing curve parameters ...\n");
    
    irand(seed);
    
    ecurve2_init(m[0],m[1],m[2],m[3],a,b,FALSE,MR_PROJECTIVE);  /* initialise curve */

    g=epoint2_init();
    w=epoint2_init();

    if (!epoint2_set(x,y,0,g)) /* initialise point of order q */
    {
        printf("Problem - point (x,y) is not on the curve\n");
        return;
    }

    ecurve2_mult(q,g,w);
    if (!point_at_infinity(w))
    {
        printf("Problem - point (x,y) is not of order q\n");
        return;
    }

/* generate public/private keys */

    bigrand(q,d);
    ecurve2_mult(d,g,g);
    
    ep=epoint2_get(g,x,x); /* compress point */

    fp=fopen("public.ecsb","w");
    fprintf(fp,"%d ",ep);
    cotnum(x,fp);
    fclose(fp);
    printf("\npublic key is:  ");
    cotnum(x,stdout);
    printf ("\nand can also be seen in public.ecsb file\n");

    fp=fopen("private.ecsb","w");
    cotnum(d,fp);
    fclose(fp);
    printf("\nprivate key is:  ");
    cotnum(d,stdout);
    printf("\nand can also be seen in private.ecsb file\n");
    return;
}

void strip(char *name)
{ /* strip off filename extension */
    int i;
    for (i=0;name[i]!='\0';i++)
    {
        if (name[i]!='.') continue;
        name[i]='\0';
        break;
    }
} 

void hashing(FILE *fp,big hash)
{ /* compute hash function */
    char h[20];
    int ch;
    sha sh;
    shs_init(&sh);
    while ((ch=fgetc(fp))!=EOF) shs_process(&sh,ch);
    shs_hash(&sh,h);
    bytes_to_big(20,h,hash);
}

void ecsbsign(char* parameter_file, char* file_to_sign)
{
    long seed=1485;
    FILE *fp;
    int i,m[4];
    miracl *mip;
    char ifname[50];
    big a,b,q,x,y,d,r,s,k,hash,n[4];
    epoint *g;
 
    fp=fopen(parameter_file,"r");
    if (fp==NULL)
    {
        printf("parameter file does not exist\n");
        return;
    }

    mip=mirsys(100,0);
    a=mirvar(1);
    b=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    d=mirvar(0);
    r=mirvar(0);
    s=mirvar(0);
    k=mirvar(0);
    hash=mirvar(0);
    for(i=0;i<4;i++) n[i]=mirvar(0);

    mip->IOBASE=10;
    for(i=0;i<4;i++)
    {
      cinnum(n[i],fp);
      m[i]=size(n[i]);
    }
    mip->IOBASE=16;
    cinnum(b,fp);     /* curve parameter */
    cinnum(x,fp);     /* (x,y) point on curve of order q */
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);     /* order of (x,y) */
    
    fclose(fp);
    mip->IOBASE=16;
    irand(seed);

    ecurve2_init(m[0],m[1],m[2],m[3],a,b,FALSE,MR_PROJECTIVE);  /* initialise curve */

    g=epoint2_init();
    epoint2_set(x,y,0,g); /* set point of order q */

    bigrand(q,k);
    ecurve2_mult(k,g,g);    
    epoint2_get(g,r,r);
    divide(r,q,q);

/* get private key of signer */
    fp=fopen("private.ecsb","r");
    if (fp==NULL)
    {
        printf("file private.ecsb does not exist\n");
        return;
    }
    cinnum(d,fp);
    fclose(fp);

/* calculate message digest */
   
    strcpy(ifname,file_to_sign);
    strip(ifname);
    strcat(ifname,".ecsb");
    if ((fp=fopen(file_to_sign,"rb"))==NULL)
    {
        printf("Unable to open file %s\n",file_to_sign);
        return;
    }
    
    printf("\nsigning file %s ...\n",file_to_sign);
    
    hashing(fp,hash);
    fclose(fp);

/* calculate s */
    xgcd(k,q,k,k,k);

    mad(d,r,hash,q,q,s);
    mad(s,k,k,q,q,s);
    fp=fopen(ifname,"w");
    cotnum(r,fp);
    cotnum(s,fp);
    fclose(fp);
    printf("\nsigned file is %s\n", ifname);
    return;
}


void ecsbverify(char* parameter_file, char* signed_file)
{
    FILE *fp;
    int ep,m[4],i;
    miracl *mip;
    epoint *g,*public;
    char ifname[50];
    big a,b,q,x,y,v,u1,u2,r,s,hash,n[4];

    fp=fopen(parameter_file,"r");
    if (fp==NULL)
    {
        printf("parameter file does not exist\n");
        return;
    }
   

    mip=mirsys(100,0);
    a=mirvar(1);
    b=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    v=mirvar(0);
    u1=mirvar(0);
    u2=mirvar(0);
    s=mirvar(0);
    r=mirvar(0);
    hash=mirvar(0);
    for(i=0;i<4;i++) n[i]=mirvar(0);
   
    mip->IOBASE=10; 
    for(i=0;i<4;i++)
    {
      cinnum(n[i],fp);
      m[i]=size(n[i]);
    }
    mip->IOBASE=16;
    cinnum(b,fp);
    cinnum(x,fp);
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);
    mip->IOBASE=16;

    
    fclose(fp);

    ecurve2_init(m[0],m[1],m[2],m[3],a,b,FALSE,MR_PROJECTIVE);  /* initialise curve */
    g=epoint2_init();
    epoint2_set(x,y,0,g); /* initialise point of order q */

/* get public key of signer */
    fp=fopen("public.ecsb","r");
    if (fp==NULL)
    {
        printf("file public.ecsb does not exist\n");
        return;
    }
    fscanf(fp,"%d",&ep);
    cinnum(x,fp);
    fclose(fp);

    public=epoint2_init();
    epoint2_set(x,x,ep,public);  /* decompress */

/* get message */
    
    strcpy(ifname,signed_file);
    strip(ifname);
    strcat(ifname,".ecsb");
    if ((fp=fopen(signed_file,"rb"))==NULL)
    { /* no message */
        printf("Unable to open file %s\n",signed_file);
        return;
    }
    hashing(fp,hash);
    fclose(fp);
    fp=fopen(ifname,"r");
    if (fp==NULL)
    { /* no signature */
        printf("signature file %s does not exist\n",ifname);
        return;
    }
    
    printf("\nverifying signature of file %s\n",signed_file);
    cinnum(r,fp);
    cinnum(s,fp);
    fclose(fp);
    if (compare(r,q)>=0 || compare(s,q)>=0)
    {
        printf("\nsignature is NOT verified\n");
        return;
    }
    xgcd(s,q,s,s,s);
    mad(hash,s,s,q,q,u1);
    mad(r,s,s,q,q,u2);

    ecurve2_mult2(u2,public,u1,g,g);
    epoint2_get(g,v,v);
    divide(v,q,q);
    if (compare(v,r)==0) printf("\nsignature is verified\n");
    else                 printf("\nsignature is NOT verified\n");
    return;
}


int main(int argc, char** argv) {
  char* file1=argv[1];
  char* file2=argv[2];
  if (argc!=3) {
    printf ("Number of arguments incorrect!\n");
    return 1;
  }
  ecsbkeygen(file1);
  ecsbsign(file1, file2);
  ecsbverify(file1,file2);
  return 0;
}

