
/*
 *   GF(p) Elliptic Curve Digital Signature Algorithm (ECDSA)
 *
 *   This program generates one set of public and private keys in files 
 *   public.ecsp and private.ecsp respectively. 
 *
 *   The curve is y^2 = x^3 -3x + b mod p
 *
 *   The curve parameter file is presumed to exist, and to contain the domain
 *   information {p,b,x,y,q}, where b is curve parameter, (x,y) is
 *   a point of order q, p is the prime modulus, and q is the order of the 
 *   point (x,y). In fact normally q is the prime number of points counted
 *   on the curve. 
 
 */

#include <stdio.h>
#include "miracl.h"
#include <stdlib.h>
#include <string.h>

static miracl *mip;


void ecspkeygen(char* parameter_file)
{
    long seed=132485;
    FILE *fp;
    int ep,bits;
    epoint *g,*w;
    big a,b,p,q,x,y,d;
    
    if((fp=fopen(parameter_file,"r"))==NULL)
    {
        printf("parameter file does not exist!\n");
        return;
    }
    
    printf("\nReading curve parameters from parameter file  ...\n");
    
    mip=mirsys(150,0);
    a=mirvar(0);
    b=mirvar(0);
    p=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    d=mirvar(0);

    mip->IOBASE=10;
    cinnum(p,fp);
    mip->IOBASE=16;
    cinnum(b,fp);
    cinnum(x,fp);
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);
    fclose(fp);
    mip->IOBASE=16;

    irand(seed);
    convert(-3,a);
    
    ecurve_init(a,b,p,MR_PROJECTIVE);  /* initialise curve */

    g=epoint_init();
    w=epoint_init();

    if (!epoint_set(x,y,0,g)) /* initialise point of order q */
    {
        printf("Problem - point (x,y) is not on the curve\n");
        exit(0);
    }

    ecurve_mult(q,g,w);
    if (!point_at_infinity(w))
    {
        printf("Problem - point (x,y) is not of order q\n");
        exit(0);
    }

/* generate public/private keys */

    bigrand(q,d);
    ecurve_mult(d,g,g);
    
    ep=epoint_get(g,x,x); /* compress point */

    fp=fopen("public.ecsp","w");
    fprintf(fp,"%d ",ep);
    cotnum(x,fp);
    fclose(fp);
    printf("\npublic key is:  ");
    cotnum(x,stdout);
    printf("\nand is also in public.ecsp file\n");
   

    fp=fopen("private.ecsp","w");
    cotnum(d,fp);
    fclose(fp);
    printf("\nprivate key is:  ");
    cotnum(d,stdout);
    printf("\nand is also in private.ecsp file\n");
    return;
}

void strip(char *name)
{ /* strip off filename extension */
    int i;
    for (i=0;name[i]!='\0';i++)
    {
        if (name[i]!='.') continue;
        name[i]='\0';
        break;
    }
}


void hashing(FILE *fp,big hash)
{ /* compute hash function */
    char h[20];
    int ch;
    sha sh;
    shs_init(&sh);
    while ((ch=fgetc(fp))!=EOF) shs_process(&sh,ch);
    shs_hash(&sh,h);
    bytes_to_big(20,h,hash);
}


void ecspsign(char* parameter_file, char* file_to_sign)
{
    long seed=11985;
    FILE *fp;
    char ifname[50];
    big a,b,p,q,x,y,d,r,s,k,hash;
    epoint *g;
    int bits;

    fp=fopen(parameter_file,"r");
    if (fp==NULL)
    {
        printf("parameter file does not exist\n");
        return;
    }

    mip=mirsys(150,0);
    a=mirvar(0);
    b=mirvar(0);
    p=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    d=mirvar(0);
    r=mirvar(0);
    s=mirvar(0);
    k=mirvar(0);
    hash=mirvar(0);

    mip->IOBASE=10;
    cinnum(p,fp); 
    mip->IOBASE=16;   
    cinnum(b,fp);
    cinnum(x,fp);     /* (x,y) point on curve of order q */
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);     /* order of (x,y) */
    fclose(fp);
    mip->IOBASE=16;
    
    irand(seed);
    convert(-3,a);
     
    ecurve_init(a,b,p,MR_PROJECTIVE);  /* initialise curve */
    g=epoint_init();
    epoint_set(x,y,0,g); /* initialise point of order q */


    bigrand(q,k);
    ecurve_mult(k,g,g);      
    epoint_get(g,r,r);
    divide(r,q,q);

/* get private key of signer */
    fp=fopen("private.ecsp","r");
    if (fp==NULL)
    {
        printf("file private.ecsp does not exist\n");
        return;
    }
    cinnum(d,fp);
    fclose(fp);

/* calculate message digest */
    
    strcpy(ifname,file_to_sign);
    strip(ifname);
    strcat(ifname,".ecsp");
    if ((fp=fopen(file_to_sign,"rb"))==NULL)
    {
        printf("Unable to open file %s\n",file_to_sign);
        return;
    }
    hashing(fp,hash);
    fclose(fp);
    printf("signing file %s\n",file_to_sign);
/* calculate s */
    xgcd(k,q,k,k,k);

    mad(d,r,hash,q,q,s);
    mad(s,k,k,q,q,s);
    fp=fopen(ifname,"w");
    cotnum(r,fp);
    cotnum(s,fp);
    printf("the signature is in the file %s\n",ifname);
    fclose(fp);
    return;
}

void ecspverify(char* parameter_file, char* signed_file)
{
    FILE *fp;
    int bits,ep;
    epoint *g,*public;
    char ifname[50];
    big a,b,p,q,x,y,v,u1,u2,r,s,hash;

/* get public data */
    fp=fopen(parameter_file,"r");
    if (fp==NULL)
    {
        printf("parameter file does not exist\n");
        return;
    }

    mip=mirsys(150,0);
    a=mirvar(0);
    b=mirvar(0);
    p=mirvar(0);
    q=mirvar(0);
    x=mirvar(0);
    y=mirvar(0);
    v=mirvar(0);
    u1=mirvar(0);
    u2=mirvar(0);
    s=mirvar(0);
    r=mirvar(0);
    hash=mirvar(0);

    mip->IOBASE=10;
    cinnum(p,fp);
    mip->IOBASE=16;
    cinnum(b,fp);
    cinnum(x,fp);
    cinnum(y,fp);
    mip->IOBASE=10;
    cinnum(q,fp);
    fclose(fp);
    mip->IOBASE=16;
    
    convert(-3,a);
    ecurve_init(a,b,p,MR_PROJECTIVE);  /* initialise curve */
    
    g=epoint_init();
    epoint_set(x,y,0,g); /* initialise point of order q */

/* get public key of signer */
    fp=fopen("public.ecsp","r");
    if (fp==NULL)
    {
        printf("file public.ecsp does not exist\n");
        return;
    }
    fscanf(fp,"%d",&ep);
    cinnum(x,fp);
    fclose(fp);

    public=epoint_init();
    epoint_set(x,x,ep,public);  /* decompress */


    strcpy(ifname,signed_file);
    strip(ifname);
    strcat(ifname,".ecsp");
    if ((fp=fopen(signed_file,"rb"))==NULL)
    { /* no message */
        printf("Unable to open file %s\n",ifname);
        return;
    }
    hashing(fp,hash);
    fclose(fp);
    fp=fopen(ifname,"r");
    if (fp==NULL)
    { /* no signature */
        printf("signature file %s does not exist\n",ifname);
        return;
    }
    cinnum(r,fp);
    cinnum(s,fp);
    fclose(fp);
    if (compare(r,q)>=0 || compare(s,q)>=0)
    {
        printf("\nsignature is NOT verified\n");
        return;
    }
    xgcd(s,q,s,s,s);
    mad(hash,s,s,q,q,u1);
    mad(r,s,s,q,q,u2);

    ecurve_mult2(u2,public,u1,g,g);
    epoint_get(g,v,v);
    divide(v,q,q);
    if (compare(v,r)==0) printf("\nsignature is verified\n");
    else                 printf("\nsignature is NOT verified\n");
    return;
}

int main (int argc, char** argv) {
  char* file1=argv[1];
  char* file2=argv[2];
  if (argc!=3) {
    printf ("Number of arguments incorrect!\n");
    return 1;
  }
  ecspkeygen(file1);
  ecspsign(file1,file2);
  ecspverify(file1,file2);
  return 0;
  }
