/*201220 Roberto Giorgi - University of Siena, Italy
##########################################################################
# MIT LICENSE:
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHERi
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
##########################################################################*/
#include <stdio.h>
#include <stdlib.h>
#include <mpi.h>

#define HISTOGRAM_BIN_COUNT 4
char* Color;
int* Histogram;
#define N 32

#define MASTER 0
#define TAG_GENERAL 1
void histo_mpi(int *histogram, char *color, int size, int wid, int nw) {

    // Per-rank fixed chunk size for MPI_Scatter
    int ssize = (size + nw - 1) / nw;          // ceil(size/nw)
    if (ssize == 0) return;
    int local_n = 0, start = wid * ssize;
    if (start < size) {
        int remaining = size - start;
        local_n = remaining < ssize ? remaining : ssize;
    }

    // Root builds a padded buffer of nw*ssize, others don't use sendbuf
    char *sendbuf = NULL;
    if (wid == MASTER) {
        int padded = nw * ssize;
        sendbuf = (char *)malloc((size_t)padded);
        // copy original data
        for (int i = 0; i < size; ++i) sendbuf[i] = color[i];
        // pad the tail with any valid bin value (0 is fine); we won't count them
        for (int i = size; i < padded; ++i) sendbuf[i] = 0;
    }

    // Each rank receives exactly ssize elements (may be zero)
    char *slice = (char *)malloc((size_t)ssize);

    MPI_Scatter(sendbuf, ssize, MPI_CHAR, slice, ssize, MPI_CHAR,
                MASTER, MPI_COMM_WORLD);

    if (wid == MASTER && sendbuf) free(sendbuf);

    // Local histogram over only the valid part of the slice
    int histogram_private[HISTOGRAM_BIN_COUNT];
    for (int i = 0; i < HISTOGRAM_BIN_COUNT; ++i) histogram_private[i] = 0;

    for (int i = 0; i < local_n; ++i) {
        unsigned int bin = (unsigned char)slice[i];
        if (bin < HISTOGRAM_BIN_COUNT) histogram_private[bin]++;
        // else ignore out-of-range values
    }
    if (slice) free(slice);

    // Single reduction into MASTER's 'histogram' buffer
    // (recvbuf is ignored on non-master ranks, so passing 'histogram' is safe there)
    MPI_Reduce(histogram_private, histogram, HISTOGRAM_BIN_COUNT,
               MPI_INT, MPI_SUM, MASTER, MPI_COMM_WORLD);
}

int main(int argc, char **argv) {
    // Initializing MPI environment
    int NumProcesses;
    int Rank;
    MPI_Init(&argc,&argv);

    // Gets number of tasks/processes that this program is running on
    MPI_Comm_size(MPI_COMM_WORLD, &NumProcesses);

    // Gets the rank (process/task number) that this program is running on
    MPI_Comm_rank(MPI_COMM_WORLD, &Rank);
    
    // Print initial data
    if (Rank == MASTER) {
        Color = (char*)malloc(N * sizeof(char));
        Histogram = (int*)malloc(HISTOGRAM_BIN_COUNT * sizeof(int));
        srand(2017);
        for (int i = 0; i < N; ++i) Color[i] = (char)(rand() % HISTOGRAM_BIN_COUNT);
        for (int i = 0; i < HISTOGRAM_BIN_COUNT; ++i) Histogram[i] = 0;
        for(int i=0; i<N; i++) printf("%3d ", Color[i]); printf("\n"); fflush(stdout);
    }

    ///////////////////////////////
    // Call the processing function
    histo_mpi(Histogram, Color, N, Rank, NumProcesses);
    ///////////////////////////////

    // Print final data
    if (Rank == MASTER) {
        printf("---\n"); fflush(stdout);
        for(int c=0; c<HISTOGRAM_BIN_COUNT; c++)
            printf("c=%3d  d=%d\n", c, Histogram[c]); fflush(stdout);
        free(Color); free(Histogram);
    }

    // Shut down MPI
    MPI_Finalize();
}
